<?php

namespace App\Http\Controllers\Talenta;

use App\Http\Controllers\Controller;
use App\Services\MekariTalentaSandboxClient;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;

class SandboxEmployeeController extends Controller
{
    private MekariTalentaSandboxClient $talenta;
    private int $companyId = 2627; // SANDBOX kamu (branch: id=0, name="Pusat")

    public function __construct(MekariTalentaSandboxClient $talenta)
    {
        $this->talenta = $talenta;
    }

    /* ================= Helpers ================= */

    private function safeData(string $path, array $query = []): array
    {
        try {
            $resp = $this->talenta->get($path, $query);
            return (is_array($resp) && isset($resp['data']) && is_array($resp['data'])) ? $resp['data'] : [];
        } catch (\Throwable $e) {
            Log::error('[SANDBOX] safeData', ['path'=>$path,'query'=>$query,'err'=>$e->getMessage()]);
            return [];
        }
    }

    private function looksLikeList(array $arr): bool
    {
        if ($arr === []) return true;
        $keys = array_keys($arr);
        $isSeq = ($keys === range(0, count($keys)-1));
        if ($isSeq) return true;
        foreach ($arr as $v) if (!is_array($v)) return false;
        return count($arr) > 1;
    }

    private function extractList(array $raw, array $cand = []): array
    {
        foreach ($cand as $k) if (isset($raw[$k]) && is_array($raw[$k])) return $raw[$k];
        foreach (['items','list','rows','result','data'] as $k) if (isset($raw[$k]) && is_array($raw[$k])) return $raw[$k];
        if ($this->looksLikeList($raw)) return $raw;
        foreach ($raw as $v) if (is_array($v) && $this->looksLikeList($v)) return $v;
        return [];
    }

    private function fetchAllPages(string $path, array $queryBase = [], int $limit = 200, array $wrapperKeys = []): array
    {
        $all = []; $page = 1;
        while (true) {
            $q = array_merge($queryBase, ['limit'=>$limit,'page'=>$page]);
            $data = $this->safeData($path, $q);
            $rows = $this->extractList($data, $wrapperKeys);
            if (!$rows) break;

            $all = array_merge($all, $rows);

            $pg = $data['pagination'] ?? ($data['meta']['pagination'] ?? null);
            $hasMore = false;
            if (is_array($pg)) {
                $cur = (int)($pg['current_page'] ?? $page);
                $tot = (int)($pg['total_pages'] ?? $page);
                $hasMore = $cur < $tot;
            } else {
                $hasMore = count($rows) >= $limit;
            }
            if (!$hasMore) break;
            if (++$page > 200) break;
        }
        return $all;
    }

    private function normalizeDateYmd(?string $val): ?string
    {
        if (!$val) return null;
        $v = trim($val);
        if (preg_match('~^\d{2}/\d{2}/\d{4}$~', $v)) { [$m,$d,$y] = explode('/', $v); return sprintf('%04d-%02d-%02d', $y,$m,$d); }
        if (preg_match('~^\d{4}-\d{2}-\d{2}$~', $v)) return $v;
        return $v;
    }

    /* ===== Masters (scoped) ===== */

    private function getBranches(): array
    {
        $raw = $this->fetchAllPages("/v2/talenta/v2/company/{$this->companyId}/branch", [], 200, ['branches','branch']);
        $out = [];
        foreach ($raw as $r) {
            if (!is_array($r)) continue;
            $id   = $r['id'] ?? $r['branch_id'] ?? null;
            $name = $r['name'] ?? $r['branch'] ?? $r['branch_name'] ?? null;
            if ($name !== null) $out[] = ['id'=> (string)($id ?? '0'), 'name'=> (string)$name];
        }
        return $out;
    }

    private function getOrganizations(): array
    {
        $raw = $this->fetchAllPages("/v2/talenta/v2/company/{$this->companyId}/organization", [], 200, ['organizations','organization']);
        $out = [];
        foreach ($raw as $r) {
            if (!is_array($r)) continue;
            $id   = $r['organization_id'] ?? $r['id'] ?? null;
            $name = $r['organization_name'] ?? $r['name'] ?? null;
            if ($name !== null) $out[] = ['id'=>(string)($id ?? ''), 'name'=>(string)$name];
        }
        return $out;
    }

    private function getJobPositions(): array
    {
        $raw = $this->fetchAllPages("/v2/talenta/v2/company/{$this->companyId}/job-position", [], 200, ['job_positions','job_position']);
        $out = [];
        foreach ($raw as $r) {
            if (!is_array($r)) continue;
            $id   = $r['job_position_id'] ?? $r['id'] ?? null;
            $name = $r['job_position'] ?? $r['job_position_name'] ?? $r['name'] ?? null;
            if ($name !== null) $out[] = ['id'=>(string)($id ?? ''), 'name'=>(string)$name];
        }
        return $out;
    }

    private function getJobLevels(): array
    {
        $paths = [
            "/v2/talenta/v2/company/{$this->companyId}/job-level",
            "/v2/talenta/v2/company/me/job-level",
            "/v2/talenta/v2/job-level?company_id={$this->companyId}",
        ];
        $by=[];
        foreach ($paths as $p) {
            $rows = $this->fetchAllPages($p, [], 200, ['job_levels','job_level','levels']);
            foreach ($rows as $r) {
                if (!is_array($r)) continue;
                $id   = $r['job_level_id'] ?? $r['id'] ?? null;
                $name = $r['job_level'] ?? $r['job_level_name'] ?? $r['name'] ?? null;
                if ($name) $by[(string)($id ?? $name)] = ['id'=>(string)($id ?? $name), 'name'=>(string)$name];
            }
        }
        $levels = array_values($by);
        if (!count($levels)) {
            // fallback: culik dari list employee
            $levels = [];
            $seen=[]; $page=1; $limit=200;
            while (true) {
                $data = $this->safeData('/v2/talenta/v2/employee', ['limit'=>$limit,'page'=>$page]);
                $list = $this->extractList($data, ['employees','employee']);
                if (!$list) break;
                foreach ($list as $emp) {
                    $e = $emp['employment'] ?? [];
                    $nm = $e['job_level'] ?? null; $id = $e['job_level_id'] ?? null;
                    if ($nm && !isset($seen[mb_strtolower($nm)])) {
                        $seen[mb_strtolower($nm)] = true;
                        $levels[] = ['id'=>(string)($id ?? $nm), 'name'=>(string)$nm];
                    }
                }
                $pg = $data['pagination'] ?? null;
                $hasMore = is_array($pg)
                    ? (int)($pg['current_page'] ?? $page) < (int)($pg['total_pages'] ?? $page)
                    : count($list) >= $limit;
                if (!$hasMore || ++$page>50) break;
            }
        }
        return $levels;
    }

    private function getEmploymentStatuses(): array
    {
        $raw = $this->fetchAllPages("/v2/talenta/v2/company/me/employment-status", [], 200, ['employment_statuses','employment_status']);
        $out = [];
        foreach ($raw as $r) {
            if (!is_array($r)) continue;
            $id   = $r['employment_status_id'] ?? $r['id'] ?? null;
            $name = $r['employment_status'] ?? $r['employment_status_name'] ?? $r['name'] ?? (string)$id;
            if ($id !== null) $out[] = ['id'=>(string)$id, 'name'=>(string)$name];
        }
        return $out;
    }

    private function mustPick(string $field, string $input, array $opts): array
    {
        $val = trim($input);
        foreach ($opts as $o) if (mb_strtolower($o['name']) === mb_strtolower($val)) return ['id'=>$o['id'], 'name'=>$o['name']];
        foreach ($opts as $o) if ((string)$o['id'] === (string)$val) return ['id'=>$o['id'], 'name'=>$o['name']];
        $allowed = implode(', ', array_map(fn($o)=>$o['name'], $opts));
        throw ValidationException::withMessages([$field => "{$field} '{$input}' tidak ditemukan. Pilihan: {$allowed}"]);
    }

    private function mapGender(mixed $v): ?int
    { if (is_numeric($v)) return (int)$v; return match(strtolower((string)$v)){ 'male','pria','laki','laki-laki'=>1, 'female','wanita','perempuan'=>2, default=>null }; }
    private function mapMarital(mixed $v): ?int
    { if (is_numeric($v)) return (int)$v; return match(strtolower((string)$v)){ 'single'=>1,'married'=>2,'widow'=>3,'widower'=>4, default=>null }; }
    private function mapReligion(mixed $v): ?int
    { if (is_numeric($v)) return (int)$v; return match(strtolower((string)$v)){ 'islam'=>1,'christian'=>2,'catholic'=>3,'buddha'=>4,'hindu'=>5,'confucius','konghucu'=>6,'others','other'=>7, default=>null }; }
    private function mapPtkp(mixed $v): ?int
    { if (is_numeric($v)) return (int)$v; return match(strtoupper((string)$v)){ 'TK/0'=>1,'TK/1'=>2,'TK/2'=>3,'TK/3'=>4,'K/0'=>5,'K/1'=>6,'K/2'=>7,'K/3'=>8, default=>null }; }
    private function mapTaxConfig(mixed $v): int
    { if (is_numeric($v)) return (int)$v; return match(strtolower((string)$v)){ 'gross'=>1,'net'=>2,'gross_up','gross-up'=>3, default=>1 }; }
    private function mapTypeSalary(mixed $v): int
    { if (is_numeric($v)) return (int)$v; return match(strtolower((string)$v)){ 'monthly'=>1,'daily'=>2, default=>1 }; }
    private function mapSalaryConfig(mixed $v): int
    { if (is_numeric($v)) return (int)$v; return match(strtolower((string)$v)){ 'gross'=>1,'net'=>2, default=>1 }; }
    private function mapYesNo0to3(mixed $v): int
    { if (is_numeric($v)) return (int)$v; return match(strtolower((string)$v)){ 'none','no','tidak'=>0,'by employee','employee'=>1,'by company','company'=>2,'both','keduanya'=>3, default=>0 }; }
    private function mapOvertime(mixed $v): int
    { if (is_numeric($v)) return (int)$v; return match(strtolower((string)$v)){ 'yes','ya'=>1,'no','tidak'=>2, default=>2 }; }
    private function mapBpjsClass(mixed $v): int
    { if (is_numeric($v)) return (int)$v; return match(strtolower((string)$v)){ '1','class_1','kelas_1'=>1,'2','class_2','kelas_2'=>2,'3','class_3','kelas_3'=>3, default=>1 }; }

    /* ===== Views ===== */

    public function index(Request $req)
    {
        $limit = $req->get('limit', 10);
        $page  = $req->get('page', 1);
        try {
            $resp   = $this->talenta->get('/v2/talenta/v2/employee', ['limit'=>$limit,'page'=>$page]);
            $items  = is_array($resp) ? data_get($resp,'data.employees',[]) : [];
            $paging = is_array($resp) ? data_get($resp,'data.pagination',[]) : [];
            return view('talenta.sandbox.employees.index', compact('items','paging','limit','page','resp'));
        } catch (\Throwable $e) {
            Log::error('[SANDBOX] index', ['e'=>$e->getMessage()]);
            $items=$paging=[]; $resp=['error'=>$e->getMessage()];
            return view('talenta.sandbox.employees.index', compact('items','paging','limit','page','resp'))
                ->withErrors(['sandbox'=>$e->getMessage()]);
        }
    }

    public function create()
    {
        try {
            $branches = $this->getBranches();
            $orgs     = $this->getOrganizations();
            $jobs     = $this->getJobPositions();
            $levels   = $this->getJobLevels();
            $emps     = $this->getEmploymentStatuses();

            $genders = [['id'=>'Male','name'=>'Male'],['id'=>'Female','name'=>'Female']];
            $maritals = [['id'=>'Single','name'=>'Single'],['id'=>'Married','name'=>'Married'],['id'=>'Widow','name'=>'Widow'],['id'=>'Widower','name'=>'Widower']];
            $religs = [
                ['id'=>'Islam','name'=>'Islam'],['id'=>'Christian','name'=>'Christian'],['id'=>'Catholic','name'=>'Catholic'],
                ['id'=>'Buddha','name'=>'Buddha'],['id'=>'Hindu','name'=>'Hindu'],['id'=>'Confucius','name'=>'Confucius'],['id'=>'Others','name'=>'Others'],
            ];
            $ptkps = [
                ['id'=>'TK/0','name'=>'TK/0'],['id'=>'TK/1','name'=>'TK/1'],['id'=>'TK/2','name'=>'TK/2'],['id'=>'TK/3','name'=>'TK/3'],
                ['id'=>'K/0','name'=>'K/0'],['id'=>'K/1','name'=>'K/1'],['id'=>'K/2','name'=>'K/2'],['id'=>'K/3','name'=>'K/3'],
            ];
            $schedules = [['id'=>'GENERAL','name'=>'GENERAL'],['id'=>'No schedule','name'=>'No schedule'],['id'=>'standart','name'=>'standart'],['id'=>'Long shift','name'=>'Long shift']];

            return view('talenta.sandbox.employees.create', compact(
                'branches','orgs','jobs','levels','emps','genders','maritals','religs','ptkps','schedules'
            ));
        } catch (\Throwable $e) {
            return back()->withErrors(['sandbox'=>$e->getMessage()]);
        }
    }

    /* ===== Payload & Sender ===== */

    private function buildBasePayload(array $input, array $branch, array $org, array $job, string $jobLevelName, string $employmentStatusId): array
    {
        // gender wajib: kalau tidak diisi, fail validasi di sini supaya tidak nabrak API
        $genderCode = $this->mapGender($input['gender'] ?? null);
        if ($genderCode === null) {
            throw ValidationException::withMessages(['gender' => 'Gender wajib diisi.']);
        }

        $payload = [
            'employee_id'   => (string)($input['employee_id'] ?? ''),
            'first_name'    => (string)($input['first_name'] ?? ''),
            'last_name'     => (string)($input['last_name'] ?? ''),
            'email'         => (string)($input['email'] ?? ''),
            'date_of_birth' => $this->normalizeDateYmd($input['date_of_birth'] ?? null),

            'gender'             => $genderCode, // <-- WAJIB

            'marital_status'     => $this->mapMarital($input['marital_status'] ?? null) ?? 1,
            'religion'           => $this->mapReligion($input['religion'] ?? null) ?? 1,

            'branch'             => $branch['name'],
            'branch_id'          => is_numeric($branch['id']) ? (int)$branch['id'] : 0,
            'branch_name'        => $branch['name'],

            'organization_name'  => $org['name'],
            'job_position'       => $job['name'],
            'job_level'          => $jobLevelName,
            'employment_status'  => (string)$employmentStatusId,
            'join_date'          => $this->normalizeDateYmd($input['join_date'] ?? null),
            'schedule'           => (string)($input['schedule'] ?? ''),

            'basic_salary'       => (int)($input['basic_salary'] ?? 0),
            'ptkp_status'        => $this->mapPtkp($input['ptkp_status'] ?? null) ?? 1,

            'tax_configuration'    => $this->mapTaxConfig($input['tax_configuration'] ?? 2),
            'type_salary'          => $this->mapTypeSalary($input['type_salary'] ?? 1),
            'salary_configuration' => $this->mapSalaryConfig($input['salary_configuration'] ?? 2),
            'jht_configuration'    => $this->mapYesNo0to3($input['jht_configuration'] ?? 1),
            'employee_tax_status'  => (int)($input['employee_tax_status'] ?? 1),
            'jp_configuration'     => $this->mapYesNo0to3($input['jp_configuration'] ?? 3),
            'overtime_status'      => $this->mapOvertime($input['overtime_status'] ?? 1),
            'bpjs_kesehatan_config'=> $this->mapBpjsClass($input['bpjs_kesehatan_config'] ?? 1),

            'npp_bpjs_ketenagakerjaan' => (string)($input['npp_bpjs_ketenagakerjaan'] ?? 'default') ?: 'default',
        ]

        ;

        // Contract (2) must end date
        if ((string)$employmentStatusId === '2') {
            $end = $this->normalizeDateYmd($input['end_employment_status_date'] ?? null);
            if (!$end) throw ValidationException::withMessages(['end_employment_status_date'=>'Wajib untuk status Contract (ID=2).']);
            $payload['end_employment_status_date'] = $end;
        } elseif (!empty($input['end_employment_status_date'])) {
            $payload['end_employment_status_date'] = $this->normalizeDateYmd($input['end_employment_status_date']);
        }

        // minimal checks
        foreach ([
            'employee_id','first_name','email','date_of_birth','join_date','gender',
            'organization_name','job_position','job_level','employment_status','basic_salary','ptkp_status'
        ] as $f) {
            if (!isset($payload[$f]) || $payload[$f]==='' || $payload[$f]===null) {
                throw ValidationException::withMessages([$f => "{$f} wajib diisi."]);
            }
        }

        return $payload;
    }

    private function sendWithFallbacks(array $payloadBase, array $branch)
    {
        // Fokus: endpoint global + query company_id. Dua mode branch: nama & id-string.
        $trialOptions = [
            ['path'=>'/v2/talenta/v2/employee', 'query'=>['company_id'=>$this->companyId], 'branch_mode'=>'name'],
            ['path'=>'/v2/talenta/v2/employee', 'query'=>['company_id'=>$this->companyId], 'branch_mode'=>'id-string'],
        ];

        $errors = [];
        foreach ($trialOptions as $trial) {
            $payload = $payloadBase;

            if ($trial['branch_mode'] === 'name') {
                $payload['branch']      = $branch['name'];             // "Pusat"
                $payload['branch_id']   = is_numeric($branch['id']) ? (int)$branch['id'] : 0; // 0
                $payload['branch_name'] = $branch['name'];             // "Pusat"
            } else { // id-string
                $payload['branch']      = (string)$branch['id'];       // "0"
                $payload['branch_id']   = is_numeric($branch['id']) ? (int)$branch['id'] : 0;
                $payload['branch_name'] = $branch['name'];
            }

            Log::debug('[TRIAL] POST', [
                'path'=>$trial['path'], 'query'=>$trial['query'],
                'branch'=>$payload['branch'],
                'branch_id'=>$payload['branch_id'],
                'branch_name'=>$payload['branch_name'],
            ]);

            try {
                return $this->talenta->post($trial['path'], $payload, $trial['query']);
            } catch (\Throwable $e) {
                $msg = $e->getMessage();
                $errors[] = ['trial'=>$trial, 'error'=>$msg];
                Log::error('[TRIAL FAILED]', ['trial'=>$trial, 'error'=>$msg]);
            }
        }

        $pack = json_encode($errors, JSON_PRETTY_PRINT);
        throw new \RuntimeException("Semua trial POST gagal. Detail:\n{$pack}");
    }

    public function store(Request $request)
    {
        try {
            // VALIDASI CEPAT (agar error di form, bukan dari API)
            $request->validate([
                'employee_id'   => 'required|string',
                'first_name'    => 'required|string',
                'email'         => 'required|email',
                'date_of_birth' => 'required|date',
                'gender'        => 'required|string', // wajib diisi
                'marital_status'=> 'required|string',
                'religion'      => 'required|string',
                'branch'        => 'required|string',
                'organization_name' => 'required|string',
                'job_position'  => 'required|string',
                'job_level'     => 'required|string',
                'employment_status' => 'required',
                'join_date'     => 'required|date',
                'basic_salary'  => 'required|integer|min:0',
                'ptkp_status'   => 'required|string',
            ]);

            // Masters & selections
            $branches = $this->getBranches();
            $orgs     = $this->getOrganizations();
            $jobs     = $this->getJobPositions();
            $levels   = $this->getJobLevels();
            $emps     = $this->getEmploymentStatuses();

            $branchSel = $this->mustPick('branch', (string)($request->input('branch','')), $branches);
            $orgSel    = $this->mustPick('organization_name', (string)$request->input('organization_name',''), $orgs);
            $jobSel    = $this->mustPick('job_position', (string)$request->input('job_position',''), $jobs);

            $jobLevelInput = trim((string)$request->input('job_level',''));
            if (!count($levels)) {
                if ($jobLevelInput === '') throw ValidationException::withMessages(['job_level'=>'Wajib diisi.']);
                $jobLevelName = $jobLevelInput;
            } else {
                $lev = $this->mustPick('job_level', $jobLevelInput, $levels);
                $jobLevelName = $lev['name'];
            }

            $emp = $this->mustPick('employment_status', (string)$request->input('employment_status',''), $emps);
            $employmentStatusId = $emp['id'];

            $payloadBase = $this->buildBasePayload($request->all(), $branchSel, $orgSel, $jobSel, $jobLevelName, $employmentStatusId);

            $resp = $this->sendWithFallbacks($payloadBase, $branchSel);

            // Aman-kan redirect: kalau tidak ada user_id, balik ke index saja
            $uid = is_array($resp) ? (data_get($resp,'data.employee.user_id') ?? data_get($resp,'data.user_id')) : null;

            if ($uid) {
                return redirect()->route('talenta.sandbox.employees.show', $uid)
                    ->with('success', "Created ke SANDBOX (user_id: {$uid})");
            }

            // fallback: tidak ada uid → index
            return redirect()->route('talenta.sandbox.employees.index')
                ->with('success', 'Created ke SANDBOX, namun user_id tidak dikembalikan API. Cek daftar karyawan untuk verifikasi.');
        } catch (ValidationException $ve) {
            return back()->withErrors($ve->errors())->withInput();
        } catch (\Throwable $e) {
            Log::error('[SANDBOX] store error', ['e'=>$e->getMessage()]);
            return back()->withErrors(['sandbox'=>$e->getMessage()])->withInput();
        }
    }

    public function show(string $user)
    {
        try {
            $resp = $this->talenta->get("/v2/talenta/v2/employee/{$user}");
            $emp  = is_array($resp) ? data_get($resp,'data.employee') : null;
            if (!$emp) return redirect()->route('talenta.sandbox.employees.index')
                ->withErrors(['sandbox'=>'Employee tidak ditemukan']);
            return view('talenta.sandbox.employees.show', compact('emp','resp'));
        } catch (\Throwable $e) {
            return redirect()->route('talenta.sandbox.employees.index')
                ->withErrors(['sandbox'=>$e->getMessage()]);
        }
    }

    public function edit(string $user)
    {
        try {
            // 1) Ambil detail employee
            $resp = $this->talenta->get("/v2/talenta/v2/employee/{$user}");
            $emp  = is_array($resp) ? data_get($resp,'data.employee') : null;
            if (!$emp) {
                return redirect()->route('talenta.sandbox.employees.index')
                    ->withErrors(['sandbox' => 'Employee tidak ditemukan']);
            }

            // 2) Masters untuk dropdown
            $branches = $this->getBranches();
            $orgs     = $this->getOrganizations();
            $jobs     = $this->getJobPositions();
            $levels   = $this->getJobLevels();
            $emps     = $this->getEmploymentStatuses();

            // 3) Static lists
            $genders = [
                ['id'=>'Male','name'=>'Male'],
                ['id'=>'Female','name'=>'Female'],
            ];
            $maritals = [
                ['id'=>'Single','name'=>'Single'],
                ['id'=>'Married','name'=>'Married'],
                ['id'=>'Widow','name'=>'Widow'],
                ['id'=>'Widower','name'=>'Widower'],
            ];
            $religs = [
                ['id'=>'Islam','name'=>'Islam'],
                ['id'=>'Christian','name'=>'Christian'],
                ['id'=>'Catholic','name'=>'Catholic'],
                ['id'=>'Buddha','name'=>'Buddha'],
                ['id'=>'Hindu','name'=>'Hindu'],
                ['id'=>'Confucius','name'=>'Confucius'],
                ['id'=>'Others','name'=>'Others'],
            ];
            $ptkps = [
                ['id'=>'TK/0','name'=>'TK/0'],['id'=>'TK/1','name'=>'TK/1'],
                ['id'=>'TK/2','name'=>'TK/2'],['id'=>'TK/3','name'=>'TK/3'],
                ['id'=>'K/0','name'=>'K/0'],  ['id'=>'K/1','name'=>'K/1'],
                ['id'=>'K/2','name'=>'K/2'],  ['id'=>'K/3','name'=>'K/3'],
            ];
            $schedules = [
                ['id'=>'GENERAL','name'=>'GENERAL'],
                ['id'=>'No schedule','name'=>'No schedule'],
                ['id'=>'standart','name'=>'standart'],
                ['id'=>'Long shift','name'=>'Long shift'],
            ];

            // 4) Kirim ke view edit dengan $emp (biar form auto-terisi)
            return view('talenta.sandbox.employees.edit', compact(
                'user','emp','branches','orgs','jobs','levels','emps',
                'genders','maritals','religs','ptkps','schedules'
            ));
        } catch (\Throwable $e) {
            return redirect()->route('talenta.sandbox.employees.index')
                ->withErrors(['sandbox' => $e->getMessage()]);
        }
    }

    /**
     * Fallback update:
     * 1) PUT /v2/talenta/v2/employee/{user}
     * 2) POST /v2/talenta/v2/employee (body + user_id)
     * 3) POST /v2/talenta/v2/employee/{user}
     */
    private function sendUpdateWithFallbacks(string $user, array $payload)
    {
        $errors = [];

        // Selalu sertakan company_id
        $payload['company_id'] = $this->companyId;

        // Schedule rule (jangan kirim "No schedule")
        if (!empty($payload['schedule'])) {
            if (strcasecmp((string)$payload['schedule'],'No schedule') === 0) {
                unset($payload['schedule']);
            }
        } else {
            unset($payload['schedule']);
        }

        // --- Try 1: PUT /employee/{user}
        try {
            return $this->talenta->put("/v2/talenta/v2/employee/{$user}", $payload);
        } catch (\Throwable $e) {
            $errors[] = ['try'=>'PUT /employee/{user}', 'error'=>$e->getMessage()];
            Log::warning('[SANDBOX UPDATE] PUT fallback failed', end($errors));
        }

        // --- Try 2: POST /employee  (dengan user_id)
        try {
            $payload2 = $payload;
            $payload2['user_id'] = $user;
            return $this->talenta->post("/v2/talenta/v2/employee", $payload2);
        } catch (\Throwable $e) {
            $errors[] = ['try'=>'POST /employee (with user_id)', 'error'=>$e->getMessage()];
            Log::warning('[SANDBOX UPDATE] POST /employee fallback failed', end($errors));
        }

        // --- Try 3: POST /employee/{user}
        try {
            return $this->talenta->post("/v2/talenta/v2/employee/{$user}", $payload);
        } catch (\Throwable $e) {
            $errors[] = ['try'=>'POST /employee/{user}', 'error'=>$e->getMessage()];
            Log::warning('[SANDBOX UPDATE] POST /employee/{user} fallback failed', end($errors));
        }

        // Semua gagal
        throw new \RuntimeException('Update gagal di semua fallback: ' . json_encode($errors, JSON_PRETTY_PRINT));
    }

    public function update(Request $request, string $user)
    {
        // Validasi minimal (selaras dengan create supaya jelas di form)
        $request->validate([
            'employee_id'        => 'required|string',
            'first_name'         => 'required|string',
            'email'              => 'required|email',
            'date_of_birth'      => 'required|date',
            'gender'             => 'required|string',
            'marital_status'     => 'required|string',
            'religion'           => 'required|string',
            'branch'             => 'required|string',
            'organization_name'  => 'required|string',
            'job_position'       => 'required|string',
            'job_level'          => 'required|string',
            'employment_status'  => 'required',
            'join_date'          => 'required|date',
            'basic_salary'       => 'required|integer|min:0',
            'ptkp_status'        => 'required|string',
        ]);

        try {
            // Master utk mapping label->id
            $branches = $this->getBranches();
            $orgs     = $this->getOrganizations();
            $jobs     = $this->getJobPositions();
            $levels   = $this->getJobLevels();
            $emps     = $this->getEmploymentStatuses();

            $branchSel = $this->mustPick('branch', (string)($request->input('branch','')), $branches);
            $orgSel    = $this->mustPick('organization_name', (string)$request->input('organization_name',''), $orgs);
            $jobSel    = $this->mustPick('job_position', (string)$request->input('job_position',''), $jobs);

            $jobLevelInput = trim((string)$request->input('job_level',''));
            if (!count($levels)) {
                if ($jobLevelInput === '') {
                    throw \Illuminate\Validation\ValidationException::withMessages(['job_level'=>'Wajib diisi.']);
                }
                $jobLevelName = $jobLevelInput; // fallback kalau master kosong
            } else {
                $lev = $this->mustPick('job_level', $jobLevelInput, $levels);
                $jobLevelName = $lev['name'];
            }

            $emp = $this->mustPick('employment_status', (string)$request->input('employment_status',''), $emps);
            $employmentStatusId = $emp['id'];

            // Build payload dasar (re-use rule create: branch_id 0 utk Pusat, dsb)
            $payload = $this->buildBasePayload(
                $request->all(),
                $branchSel,
                $orgSel,
                $jobSel,
                $jobLevelName,
                $employmentStatusId
            );

            // Kirim dengan fallback metode/path supaya lolos dari "method not allowed"
            $this->sendUpdateWithFallbacks($user, $payload);

            return redirect()->route('talenta.sandbox.employees.show', $user)
                ->with('success', 'Updated di SANDBOX.');
        } catch (\Illuminate\Validation\ValidationException $ve) {
            return back()->withErrors($ve->errors())->withInput();
        } catch (\Throwable $e) {
            Log::error('[SANDBOX] update error', ['user'=>$user, 'e'=>$e->getMessage()]);
            return back()->withErrors(['sandbox'=>$e->getMessage()])->withInput();
        }
    }

    public function destroy(string $user)
    {
        try {
            $this->talenta->delete("/v2/talenta/v2/employee/{$user}");
            return redirect()->route('talenta.sandbox.employees.index')
                ->with('success', 'Employee dihapus (SANDBOX).');
        } catch (\Throwable $e) {
            Log::error('[SANDBOX] destroy error', ['user'=>$user, 'e'=>$e->getMessage()]);
            return back()->withErrors(['sandbox'=>$e->getMessage()]);
        }
    }

}
